<?php
require __DIR__ . '/helper.php';
$servers_raw = include __DIR__ . '/servers.php';
$servers = getServersWithActualExpire($servers_raw);
?>
<!DOCTYPE html>
<html lang="zh">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title>宝塔面板 - 服务器监控 v1.8</title>
<style>
* { box-sizing: border-box; }
body {
  background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
  font-family: "Microsoft YaHei", "Segoe UI", sans-serif;
  transition: background 0.3s;
  margin: 0;
  padding: 0;
  min-height: 100vh;
}
.container {
  width: 96%;
  max-width: 1600px;
  margin: 0 auto;
  padding: 20px 0;
}
.header {
  background: rgba(255,255,255,0.95);
  backdrop-filter: blur(10px);
  padding: 20px 30px;
  border-radius: 12px;
  box-shadow: 0 8px 32px rgba(0,0,0,0.1);
  margin-bottom: 20px;
  display: flex;
  justify-content: space-between;
  align-items: center;
  flex-wrap: wrap;
  gap: 15px;
}
h1 {
  font-size: 24px;
  margin: 0;
  color: #2c3e50;
  font-weight: 600;
}
.subtitle {
  font-size: 13px;
  color: #7f8c8d;
  margin-top: 5px;
}
.toggle {
  cursor: pointer;
  padding: 10px 20px;
  border-radius: 25px;
  background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
  color: white;
  transition: all 0.3s;
  border: none;
  font-size: 14px;
  font-weight: 500;
  box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
}
.toggle:hover {
  transform: translateY(-2px);
  box-shadow: 0 6px 20px rgba(102, 126, 234, 0.6);
}
.table-container {
  background: rgba(255,255,255,0.95);
  backdrop-filter: blur(10px);
  border-radius: 12px;
  box-shadow: 0 8px 32px rgba(0,0,0,0.1);
  overflow: hidden;
}
table {
  width: 100%;
  border-collapse: collapse;
  table-layout: fixed;
}
/* 固定列宽防止跳动 */
th:nth-child(1), td:nth-child(1) { width: 12%; }  /* 服务器 */
th:nth-child(2), td:nth-child(2) { width: 10%; }  /* 到期 */
th:nth-child(3), td:nth-child(3) { width: 8%; }   /* 价格 */
th:nth-child(4), td:nth-child(4) { width: 11%; }  /* CPU */
th:nth-child(5), td:nth-child(5) { width: 11%; }  /* 内存 */
th:nth-child(6), td:nth-child(6) { width: 22%; }  /* 硬盘 */
th:nth-child(7), td:nth-child(7) { width: 14%; }  /* 网络 */
th:nth-child(8), td:nth-child(8) { width: 12%; }  /* 版本 */

th, td {
  padding: 16px 12px;
  text-align: center;
  border-bottom: 1px solid #ecf0f1;
  font-size: 14px;
  vertical-align: middle;
  min-height: 80px;
}
th {
  background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
  color: white;
  font-weight: 600;
  text-transform: uppercase;
  font-size: 12px;
  letter-spacing: 0.5px;
  position: sticky;
  top: 0;
  z-index: 10;
}
tbody tr {
  transition: background 0.2s;
  height: 120px;
}
tbody tr:hover {
  background: rgba(102, 126, 234, 0.05);
}
tbody tr:last-child td {
  border-bottom: none;
}
.server-name-box {
  display: flex;
  flex-direction: column;
  align-items: center;
  gap: 4px;
}
.server-name {
  font-weight: 700;
  font-size: 15px;
  color: #2c3e50;
}
.server-vendor {
  font-size: 11px;
  color: #95a5a6;
  background: rgba(149, 165, 166, 0.1);
  padding: 2px 8px;
  border-radius: 10px;
  font-weight: 500;
}
.progress {
  width: 100%;
  max-width: 120px;
  height: 20px;
  background: #ecf0f1;
  border-radius: 10px;
  overflow: hidden;
  margin-top: 5px;
  box-shadow: inset 0 2px 4px rgba(0,0,0,0.1);
  position: relative;
}
.bar {
  height: 100%;
  transition: width 0.5s ease;
  border-radius: 10px;
  position: relative;
  overflow: hidden;
}
.bar::after {
  content: '';
  position: absolute;
  top: 0;
  left: 0;
  right: 0;
  bottom: 0;
  background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
  animation: shimmer 2s infinite;
}
@keyframes shimmer {
  0% { transform: translateX(-100%); }
  100% { transform: translateX(100%); }
}
.value-display {
  font-weight: 600;
  font-size: 15px;
  color: #2c3e50;
  margin-bottom: 5px;
  height: 22px;
  line-height: 22px;
}
.disk-grid {
  display: flex;
  flex-direction: column;
  gap: 8px;
  min-height: 60px;
}
.disk-part {
  text-align: left;
  padding: 6px 10px;
  background: rgba(236, 240, 241, 0.5);
  border-radius: 8px;
  transition: background 0.2s, transform 0.2s;
}
.disk-part:hover {
  background: rgba(236, 240, 241, 0.8);
  transform: translateX(3px);
}
.disk-label {
  font-weight: 600;
  font-size: 12px;
  color: #34495e;
  margin-bottom: 2px;
  height: 18px;
  line-height: 18px;
}
.disk-info {
  font-size: 11px;
  color: #7f8c8d;
  margin-bottom: 3px;
  height: 16px;
  line-height: 16px;
}
.expire-badge {
  display: inline-block;
  padding: 4px 12px;
  border-radius: 12px;
  font-weight: 600;
  font-size: 13px;
  min-width: 80px;
  height: 28px;
  line-height: 20px;
}
.expire-normal { background: #d5f4e6; color: #27ae60; }
.expire-warning { background: #ffe8cc; color: #e67e22; }
.expire-danger { background: #fadbd8; color: #e74c3c; }
.price-tag {
  font-weight: 600;
  color: #8e44ad;
  font-size: 14px;
  display: block;
  height: 22px;
  line-height: 22px;
}
.version-tag {
  font-size: 12px;
  color: #7f8c8d;
  background: rgba(149, 165, 166, 0.1);
  padding: 4px 10px;
  border-radius: 8px;
  display: inline-block;
  min-width: 60px;
  height: 26px;
  line-height: 18px;
}
.net-info {
  font-size: 13px;
  line-height: 1.8;
  min-height: 50px;
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  gap: 4px;
}
.net-row {
  display: flex;
  align-items: center;
  justify-content: center;
  width: 100%;
  height: 22px;
}
.net-icon {
  width: 20px;
  text-align: center;
  flex-shrink: 0;
}
.net-value {
  font-family: 'Consolas', 'Monaco', 'Courier New', monospace;
  font-weight: 600;
  display: inline-block;
  text-align: right;
  min-width: 100px;
  white-space: nowrap;
}
.net-up { color: #e74c3c; }
.net-down { color: #27ae60; }
.footer-note {
  text-align: center;
  margin-top: 20px;
  padding: 15px;
  background: rgba(255,255,255,0.9);
  backdrop-filter: blur(10px);
  border-radius: 12px;
  font-size: 12px;
  color: #7f8c8d;
  box-shadow: 0 4px 16px rgba(0,0,0,0.1);
}
.loading {
  color: #95a5a6;
  font-style: italic;
  height: 20px;
  line-height: 20px;
}
.error {
  color: #e74c3c;
  font-weight: 600;
  height: 20px;
  line-height: 20px;
}
.cell-wrapper {
  min-height: 60px;
  display: flex;
  flex-direction: column;
  justify-content: center;
  align-items: center;
  width: 100%;
}

/* 暗黑模式 */
body.dark {
  background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
}
body.dark .header,
body.dark .table-container,
body.dark .footer-note {
  background: rgba(30, 30, 46, 0.95);
  color: #ecf0f1;
}
body.dark h1 {
  color: #ecf0f1;
}
body.dark .subtitle {
  color: #95a5a6;
}
body.dark th, td {
  border-color: #2c3e50;
}
body.dark tbody tr:hover {
  background: rgba(102, 126, 234, 0.15);
}
body.dark .server-name {
  color: #ecf0f1;
}
body.dark .server-vendor {
  background: rgba(149, 165, 166, 0.2);
  color: #95a5a6;
}
body.dark .value-display {
  color: #ecf0f1;
}
body.dark .disk-part {
  background: rgba(52, 73, 94, 0.3);
}
body.dark .disk-part:hover {
  background: rgba(52, 73, 94, 0.5);
}
body.dark .disk-label {
  color: #ecf0f1;
}
body.dark .disk-info {
  color: #95a5a6;
}
body.dark .progress {
  background: #2c3e50;
}
body.dark .version-tag {
  background: rgba(149, 165, 166, 0.2);
  color: #95a5a6;
}

/* 响应式设计 */
@media(max-width: 1200px) {
  th, td { padding: 12px 8px; font-size: 13px; }
  .net-value { min-width: 90px; font-size: 12px; }
  .server-name { font-size: 14px; }
  .server-vendor { font-size: 10px; }
  .progress { max-width: 100px; }
}
@media(max-width: 768px) {
  .header { padding: 15px 20px; }
  h1 { font-size: 18px; }
  .table-container { overflow-x: auto; }
  table { min-width: 1000px; table-layout: auto; }
  th, td { padding: 10px 6px; font-size: 12px; }
  .net-value { min-width: 80px; font-size: 11px; }
  .server-name { font-size: 13px; }
  .server-vendor { font-size: 9px; }
  .progress { max-width: 80px; }
}
</style>
</head>
<body>
<div class="container">
  <div class="header">
    <div>
      <h1>🖥️ 宝塔面板监控中心</h1>
      <div class="subtitle">实时监控服务器状态 · 每5秒自动刷新 · 到期智能提醒 · 自动续期计算</div>
    </div>
    <div>
      <button class="toggle" onclick="toggleDark()">🌙 夜间模式</button>
    </div>
  </div>

  <div class="table-container">
    <table id="server-table">
      <thead>
        <tr>
          <th>服务器</th>
          <th>到期时间</th>
          <th>价格</th>
          <th>CPU使用率</th>
          <th>内存使用率</th>
          <th>硬盘分区</th>
          <th>网络流量</th>
          <th>面板版本</th>
        </tr>
      </thead>
      <tbody>
<?php foreach($servers as $srv): ?>
        <tr id="srv-<?= htmlspecialchars($srv['key'], ENT_QUOTES) ?>">
          <td>
            <div class="server-name-box">
              <span class="server-name"><?= htmlspecialchars($srv['name'], ENT_QUOTES) ?></span>
              <span class="server-vendor"><?= htmlspecialchars($srv['vendor'] ?? '', ENT_QUOTES) ?></span>
            </div>
          </td>
          <td class="expire" data-expire="<?= htmlspecialchars($srv['expire'], ENT_QUOTES) ?>">
            <span class="expire-badge expire-normal"><?= date('m/d', strtotime($srv['expire'])) ?></span>
          </td>
          <td><span class="price-tag"><?= htmlspecialchars($srv['price'], ENT_QUOTES) ?></span></td>
          <td class="cpu"><div class="cell-wrapper"><span class="loading">加载中...</span></div></td>
          <td class="mem"><div class="cell-wrapper"><span class="loading">加载中...</span></div></td>
          <td class="disk"><div class="cell-wrapper"><span class="loading">加载中...</span></div></td>
          <td class="net"><div class="cell-wrapper"><span class="loading">加载中...</span></div></td>
          <td class="version"><div class="cell-wrapper"><span class="loading">加载中...</span></div></td>
        </tr>
<?php endforeach; ?>
      </tbody>
    </table>
  </div>

  <div class="footer-note">
    🔒 安全提示：页面不暴露 API Key，后端使用官方签名认证并保存 Cookie · 
    🔄 到期日期自动续期计算，无需手动修改配置文件
  </div>
</div>

<script>
function toggleDark(){
  document.body.classList.toggle('dark');
  localStorage.setItem('darkMode', document.body.classList.contains('dark') ? '1' : '0');
}

window.onload = function(){
  if(localStorage.getItem('darkMode') === '1') document.body.classList.add('dark');
  refreshAll();
  setInterval(refreshAll, 5000);
}

function formatTraffic(kb){
  let value, unit;
  if(kb >= 1024*1024) {
    value = (kb / 1024 / 1024).toFixed(2);
    unit = 'GB/s';
  } else if(kb >= 1024) {
    value = (kb / 1024).toFixed(2);
    unit = 'MB/s';
  } else if(kb >= 1) {
    value = kb.toFixed(2);
    unit = 'KB/s';
  } else {
    value = (kb*1024).toFixed(2);
    unit = 'B/s';
  }
  return value.padStart(8, ' ') + ' ' + unit;
}

function refreshAll(){
  const rows = Array.from(document.querySelectorAll('tbody tr'));
  rows.forEach(row=>{
    const key = row.id.replace('srv-','');
    fetch('server_status.php?key=' + encodeURIComponent(key))
      .then(r=>r.json())
      .then(data=>{
        if(data.error){
          ['cpu','mem','disk','net','version'].forEach(c=>{
            const cell = row.querySelector('.'+c);
            if(cell) cell.innerHTML = '<div class="cell-wrapper"><span class="error">错误</span></div>';
          });
          return;
        }

        // CPU
        const cpuVal = parseFloat(data.cpu) || 0;
        const cpuColor = cpuVal > 80 ? '#e74c3c' : (cpuVal > 60 ? '#f39c12' : '#27ae60');
        const cpuCell = row.querySelector('.cpu');
        if(cpuCell) {
          cpuCell.innerHTML = `
            <div class="cell-wrapper">
              <div class="value-display">${cpuVal.toFixed(1)}%</div>
              <div class="progress">
                <div class="bar" style="width:${cpuVal}%;background:${cpuColor};"></div>
              </div>
            </div>`;
        }

        // 内存
        const memVal = parseFloat(data.mem) || 0;
        const memColor = memVal > 80 ? '#e74c3c' : (memVal > 60 ? '#f39c12' : '#3498db');
        const memCell = row.querySelector('.mem');
        if(memCell) {
          memCell.innerHTML = `
            <div class="cell-wrapper">
              <div class="value-display">${memVal.toFixed(1)}%</div>
              <div class="progress">
                <div class="bar" style="width:${memVal}%;background:${memColor};"></div>
              </div>
            </div>`;
        }

        // 磁盘
        let diskStr = '<div class="disk-grid">';
        if(data.disk && data.disk.length > 0) {
          data.disk.forEach(d=>{
            const usedVal = parseFloat(d.used) || 0;
            const diskColor = usedVal > 80 ? '#e74c3c' : (usedVal > 60 ? '#f39c12' : '#9b59b6');
            diskStr += `
              <div class="disk-part">
                <div class="disk-label">${d.path}</div>
                <div class="disk-info">${d.usedSize} / ${d.totalSize}</div>
                <div class="progress" style="height:14px;max-width:100%;">
                  <div class="bar" style="width:${usedVal}%;background:${diskColor};"></div>
                </div>
              </div>`;
          });
        }
        diskStr += '</div>';
        const diskCell = row.querySelector('.disk');
        if(diskCell) diskCell.innerHTML = diskStr;

        // 网络
        const netCell = row.querySelector('.net');
        if(netCell) {
          const upTraffic = formatTraffic(data.net_up);
          const downTraffic = formatTraffic(data.net_down);
          netCell.innerHTML = `
            <div class="net-info">
              <div class="net-row">
                <span class="net-icon">↑</span>
                <span class="net-value net-up">${upTraffic}</span>
              </div>
              <div class="net-row">
                <span class="net-icon">↓</span>
                <span class="net-value net-down">${downTraffic}</span>
              </div>
            </div>`;
        }

        // 面板版本
        const versionCell = row.querySelector('.version');
        if(versionCell) {
          versionCell.innerHTML = `<div class="cell-wrapper"><span class="version-tag">v${data.version}</span></div>`;
        }

        // 到期颜色
        const expireBadge = row.querySelector('.expire .expire-badge');
        const days = data.expire_diff;
        if(expireBadge && days !== null){
          expireBadge.classList.remove('expire-normal', 'expire-warning', 'expire-danger');
          if(days < 0) {
            expireBadge.classList.add('expire-danger');
            expireBadge.textContent = '已过期';
          } else if(days <= 7) {
            expireBadge.classList.add('expire-warning');
            expireBadge.textContent = data.expire_fmt + ' (' + days + '天)';
          } else {
            expireBadge.classList.add('expire-normal');
            expireBadge.textContent = data.expire_fmt;
          }
        }
      })
      .catch(e=>{
        ['cpu','mem','disk','net','version'].forEach(c=>{
          const cell = row.querySelector('.'+c);
          if(cell) cell.innerHTML = '<div class="cell-wrapper"><span class="error">连接失败</span></div>';
        });
      });
  });
}
</script>
</body>
</html>