<?php
/**
 * 计算实际到期日期
 * 如果当前日期已过first_expire，自动按续费周期推算到未来
 * 
 * @param string $first_expire 首次到期日期 (Y-m-d)
 * @param string $renew_cycle 续费周期: monthly(月), quarterly(季度), yearly(年)
 * @return string 实际到期日期 (Y-m-d)
 */
function calculateActualExpire($first_expire, $renew_cycle = 'monthly') {
    $first_expire_time = strtotime($first_expire);
    $current_time = time();
    
    // 如果还没到首次到期日期，直接返回
    if ($current_time < $first_expire_time) {
        return $first_expire;
    }
    
    // 已过期，需要计算续费后的日期
    $expire_time = $first_expire_time;
    
    // 根据续费周期确定时间间隔
    switch (strtolower($renew_cycle)) {
        case 'monthly':
            $interval = '+1 month';
            break;
        case 'quarterly':
            $interval = '+3 months';
            break;
        case 'yearly':
            $interval = '+1 year';
            break;
        default:
            $interval = '+1 month';
    }
    
    // 循环推算，直到找到未来的到期日期
    while ($expire_time < $current_time) {
        $expire_time = strtotime($interval, $expire_time);
    }
    
    return date('Y-m-d', $expire_time);
}

/**
 * 获取处理后的服务器列表（带实际到期日期）
 * 
 * @param array $servers 原始服务器配置数组
 * @return array 处理后的服务器数组（包含计算后的expire字段）
 */
function getServersWithActualExpire($servers) {
    foreach ($servers as &$server) {
        // 如果配置了首次到期日期和续费周期，则自动计算
        if (isset($server['first_expire']) && isset($server['renew_cycle'])) {
            $server['expire'] = calculateActualExpire($server['first_expire'], $server['renew_cycle']);
        } 
        // 如果只配置了首次到期日期，直接使用
        elseif (isset($server['first_expire'])) {
            $server['expire'] = $server['first_expire'];
        }
        // 兼容旧配置：如果直接配置了expire字段，保持不变
        elseif (!isset($server['expire'])) {
            $server['expire'] = date('Y-m-d', strtotime('+30 days')); // 默认30天后
        }
    }
    return $servers;
}